
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_ttf_conv.c */


#include "fs_itype.h"

#ifdef FS_ACT3

static FS_CONST coordEncodingType coordEncoding[128] =
{
    {2, 0, 8, 0, 0, 0, 1 },
    {2, 0, 8, 0, 0, 0, 0 },
    {2, 0, 8, 0, 256, 0, 1 },
    {2, 0, 8, 0, 256, 0, 0 },
    {2, 0, 8, 0, 512, 0, 1 },
    {2, 0, 8, 0, 512, 0, 0 },
    {2, 0, 8, 0, 768, 0, 1 },
    {2, 0, 8, 0, 768, 0, 0 },
    {2, 0, 8, 0, 1024, 0, 1 },
    {2, 0, 8, 0, 1024, 0, 0 },
    {2, 8, 0, 0, 0, 1, 0 },
    {2, 8, 0, 0, 0, 0, 0 },
    {2, 8, 0, 256, 0, 1, 0 },
    {2, 8, 0, 256, 0, 0, 0 },
    {2, 8, 0, 512, 0, 1, 0 },
    {2, 8, 0, 512, 0, 0, 0 },
    {2, 8, 0, 768, 0, 1, 0 },
    {2, 8, 0, 768, 0, 0, 0 },
    {2, 8, 0, 1024, 0, 1, 0 },
    {2, 8, 0, 1024, 0, 0, 0 },
    {2, 4, 4, 1, 1, 1, 1  },
    {2, 4, 4, 1, 1, 0, 1  },
    {2, 4, 4, 1, 1, 1, 0  },
    {2, 4, 4, 1, 1, 0, 0  },
    {2, 4, 4, 1, 17, 1, 1 },
    {2, 4, 4, 1, 17, 0, 1 },
    {2, 4, 4, 1, 17, 1, 0 },
    {2, 4, 4, 1, 17, 0, 0 },
    {2, 4, 4, 1, 33, 1, 1 },
    {2, 4, 4, 1, 33, 0, 1 },
    {2, 4, 4, 1, 33, 1, 0 },
    {2, 4, 4, 1, 33, 0, 0 },
    {2, 4, 4, 1, 49, 1, 1 },
    {2, 4, 4, 1, 49, 0, 1 },
    {2, 4, 4, 1, 49, 1, 0 },
    {2, 4, 4, 1, 49, 0, 0 },
    {2, 4, 4, 17, 1, 1, 1 },
    {2, 4, 4, 17, 1, 0, 1 },
    {2, 4, 4, 17, 1, 1, 0 },
    {2, 4, 4, 17, 1, 0, 0 },
    {2, 4, 4, 17, 17, 1, 1 },
    {2, 4, 4, 17, 17, 0, 1 },
    {2, 4, 4, 17, 17, 1, 0 },
    {2, 4, 4, 17, 17, 0, 0 },
    {2, 4, 4, 17, 33, 1, 1 },
    {2, 4, 4, 17, 33, 0, 1 },
    {2, 4, 4, 17, 33, 1, 0 },
    {2, 4, 4, 17, 33, 0, 0 },
    {2, 4, 4, 17, 49, 1, 1 },
    {2, 4, 4, 17, 49, 0, 1 },
    {2, 4, 4, 17, 49, 1, 0 },
    {2, 4, 4, 17, 49, 0, 0 },
    {2, 4, 4, 33, 1, 1, 1 },
    {2, 4, 4, 33, 1, 0, 1 },
    {2, 4, 4, 33, 1, 1, 0 },
    {2, 4, 4, 33, 1, 0, 0 },
    {2, 4, 4, 33, 17, 1, 1 },
    {2, 4, 4, 33, 17, 0, 1 },
    {2, 4, 4, 33, 17, 1, 0 },
    {2, 4, 4, 33, 17, 0, 0 },
    {2, 4, 4, 33, 33, 1, 1 },
    {2, 4, 4, 33, 33, 0, 1 },
    {2, 4, 4, 33, 33, 1, 0 },
    {2, 4, 4, 33, 33, 0, 0 },
    {2, 4, 4, 33, 49, 1, 1 },
    {2, 4, 4, 33, 49, 0, 1 },
    {2, 4, 4, 33, 49, 1, 0 },
    {2, 4, 4, 33, 49, 0, 0 },
    {2, 4, 4, 49, 1, 1, 1 },
    {2, 4, 4, 49, 1, 0, 1 },
    {2, 4, 4, 49, 1, 1, 0 },
    {2, 4, 4, 49, 1, 0, 0 },
    {2, 4, 4, 49, 17, 1, 1 },
    {2, 4, 4, 49, 17, 0, 1 },
    {2, 4, 4, 49, 17, 1, 0 },
    {2, 4, 4, 49, 17, 0, 0 },
    {2, 4, 4, 49, 33, 1, 1 },
    {2, 4, 4, 49, 33, 0, 1 },
    {2, 4, 4, 49, 33, 1, 0 },
    {2, 4, 4, 49, 33, 0, 0 },
    {2, 4, 4, 49, 49, 1, 1 },
    {2, 4, 4, 49, 49, 0, 1 },
    {2, 4, 4, 49, 49, 1, 0 },
    {2, 4, 4, 49, 49, 0, 0 },
    {3, 8, 8, 1, 1, 1, 1 },
    {3, 8, 8, 1, 1, 0, 1 },
    {3, 8, 8, 1, 1, 1, 0 },
    {3, 8, 8, 1, 1, 0, 0 },
    {3, 8, 8, 1, 257, 1, 1 },
    {3, 8, 8, 1, 257, 0, 1 },
    {3, 8, 8, 1, 257, 1, 0 },
    {3, 8, 8, 1, 257, 0, 0 },
    {3, 8, 8, 1, 513, 1, 1 },
    {3, 8, 8, 1, 513, 0, 1 },
    {3, 8, 8, 1, 513, 1, 0 },
    {3, 8, 8, 1, 513, 0, 0 },
    {3, 8, 8, 257, 1, 1, 1 },
    {3, 8, 8, 257, 1, 0, 1 },
    {3, 8, 8, 257, 1, 1, 0 },
    {3, 8, 8, 257, 1, 0, 0 },
    {3, 8, 8, 257, 257, 1, 1 },
    {3, 8, 8, 257, 257, 0, 1 },
    {3, 8, 8, 257, 257, 1, 0 },
    {3, 8, 8, 257, 257, 0, 0 },
    {3, 8, 8, 257, 513, 1, 1 },
    {3, 8, 8, 257, 513, 0, 1 },
    {3, 8, 8, 257, 513, 1, 0 },
    {3, 8, 8, 257, 513, 0, 0 },
    {3, 8, 8, 513, 1, 1, 1 },
    {3, 8, 8, 513, 1, 0, 1 },
    {3, 8, 8, 513, 1, 1, 0 },
    {3, 8, 8, 513, 1, 0, 0 },
    {3, 8, 8, 513, 257, 1, 1 },
    {3, 8, 8, 513, 257, 0, 1 },
    {3, 8, 8, 513, 257, 1, 0 },
    {3, 8, 8, 513, 257, 0, 0 },
    {3, 8, 8, 513, 513, 1, 1 },
    {3, 8, 8, 513, 513, 0, 1 },
    {3, 8, 8, 513, 513, 1, 0 },
    {3, 8, 8, 513, 513, 0, 0 },
    {4, 12, 12, 0, 0, 1, 1 },
    {4, 12, 12, 0, 0, 0, 1 },
    {4, 12, 12, 0, 0, 1, 0 },
    {4, 12, 12, 0, 0, 0, 0 },
    {5, 16, 16, 0, 0, 1, 1 },
    {5, 16, 16, 0, 0, 0, 1 },
    {5, 16, 16, 0, 0, 1, 0 },
    {5, 16, 16, 0, 0, 0, 0 }
};


/* Reads a word and increments the pointer */
/* It works for both big and little endian machines */
FS_SHORT READWORD_INC( FS_BYTE * *pRef )
{
    FS_SHORT x;
    register FS_BYTE *p = *pRef;

    x = (FS_SHORT)((((FS_SHORT) * p) << 8) | *(FS_BYTE *)(p + 1));

    *pRef += 2;
    return x; /*****/
}
FS_ULONG READLONG_INC( FS_BYTE * *pRef )
{
    FS_ULONG x;
    register FS_BYTE *p = *pRef;
    x = (FS_ULONG)((((FS_ULONG) * p) << 24) | (((FS_ULONG) * (p + 1)) << 16) | (((FS_ULONG) * (p + 2)) << 8)
                   | ((FS_ULONG) * (p + 3)));
    *pRef += 4;
    return x; /*****/
}

#define WRITEWORD_INC_MACRO( p, x ) (*(p++) = (FS_BYTE)((FS_USHORT)(x) >> 8), *(p++)    = (FS_BYTE)(x))

FS_CONST FS_BYTE flipSignCode        = 250;
FS_CONST FS_BYTE Hop3Code            = 251; /* A,X1,A,X2,A -> A,X1,Hop3Code,X2 - >A,X1,A,X2,A */
FS_CONST FS_BYTE Hop4Code            = 252; /* A,X1,A,X2,A,X3,A -> A,X1,Hop4Code,X2,X3 - >A,X1,A,X2,A,X3,A */
FS_CONST FS_BYTE wordCode            = 253;
FS_CONST FS_BYTE oneMoreByteCode2    = 254;
FS_CONST FS_BYTE oneMoreByteCode1    = 255;

FS_CONST FS_SHORT lowestUCode = 253; /* wordCode */

FS_CONST FS_SHORT lowestCode  = 250; /* flipSignCode */


/* Reads a FS_SHORT in our 255UShort format */
static FS_SHORT Read255UShort( FS_BYTE * *pRef )
{
    FS_BYTE code;
    FS_SHORT value;
    FS_BYTE *p = *pRef;

    code = *((FS_BYTE *)p++);
    if ( code == wordCode )
    {
        value = *p++;
        value <<= 8;
        value |= (FS_BYTE) * p++;
    }
    else if ( code == oneMoreByteCode1 )
    {
        value = *((FS_BYTE *)p++);
        value = (FS_SHORT)(value + lowestUCode);
    }
    else if ( code == oneMoreByteCode2 )
    {
        value = *((FS_BYTE *)p++);
        value = (FS_SHORT)(value + lowestUCode * 2);
    }
    else
    {
        value = code;
    }
    *pRef = p;
    return value; /*****/
}


/* The Glyph class */
/* This class is internal to this file */
/* and not intended to be used by clients */
/* outside of this file */
typedef struct
{
    /* private */
    FS_SHORT xmax, xmin, ymax, ymin;
    FS_SHORT leftSideBearing, advanceWidth;

    FS_SHORT numberOfContours;
    FS_SHORT numberOfPoints;

    FS_USHORT *startPoint;
    FS_USHORT *endPoint;
    FS_SHORT *x;
    FS_SHORT *y;
    FS_BYTE *onCurve;
    FS_SHORT *componentData;
    FS_SHORT maxComponentSize;
    FS_SHORT componentSize;
    FS_SHORT componentVersionNumber;
    FS_USHORT numberOfInstructions;
    FS_BYTE *code;
    FS_USHORT    pushCount;
    FS_SHORT    *pushData;
    FS_USHORT    remainingCodeSize;
    FS_BYTE    *remainingCode;

    FS_BYTE *inDataStart;        /* The start of the glyph table (ctf or ttf) */
    FS_BYTE *inDataEnd;        /* Points at the first byte past the glyph table (ctf or ttf) */

} Glyph;

/* Deconstructor */

static FS_VOID MTX_Glyph_Destroy( _DS_ Glyph* t )
{
    if (t)
    {
        FSS_free( _PS_ t->startPoint);
        FSS_free( _PS_ t->x);
        FSS_free( _PS_ t->onCurve);
        FSS_free( _PS_ t->code);
        FSS_free( _PS_ t->componentData);
        FSS_free( _PS_ t->pushData);
        FSS_free( _PS_ t->remainingCode);
        FSS_free( _PS_ t );
    }
}


static int AllocateGlyphSpace( _DS_ FS_BYTE * *destSfnt, FS_LONG pos, FS_LONG byteCount, FS_LONG *maxOutSize )
{
    byteCount += 4; /* Always keep a 4 byte margin to allow for the zero padding */
    if ( pos + byteCount > *maxOutSize )
    {
        *maxOutSize = pos + byteCount + (*maxOutSize >> 1) + 2; /* Allocate memory in exponentially increasing steps */

#ifdef FS_MEM_DBG
        STATE.memdbgid = "AllocateGlyphSpace";
#endif
        *destSfnt = (FS_BYTE *)FSS_realloc(_PS_ (FS_VOID *)*destSfnt, (FS_ULONG) * maxOutSize );
        if (STATE.error)
            return (STATE.error);
    }
    return 0;    /* success */
}



/* Computes the bounding box for this glyph */
static FS_VOID SetBBox( register Glyph *t )
{
    register FS_LONG i, limit;
    register FS_SHORT coord, *xp, *yp;

    xp = t->x;
    yp = t->y;
    t->xmax = t->xmin = xp[0];
    t->ymax = t->ymin = yp[0];
    limit = t->numberOfPoints;
    for ( i = 1; i < limit; i++ )
    {
        if ( (coord = xp[i]) > t->xmax )
        {
            t->xmax = coord;
        }
        else if ( coord < t->xmin )
        {
            t->xmin = coord;
        }
        if ( (coord = yp[i]) > t->ymax )
        {
            t->ymax = coord;
        }
        else if ( coord < t->ymin )
        {
            t->ymin = coord;
        }
    }
}

/* Writes out a glyph in the TrueType format */
static FS_LONG MTX_Glyph_WriteTTFSpline( _DS_ register Glyph *t, FS_BYTE * *destSfnt, FS_LONG pos, FS_LONG *maxOutSize )
{
    FS_BYTE *f;
    FS_SHORT i, limit;
    FS_SHORT x, y;
    FS_SHORT delta, j;
    FS_BYTE bitFlags;
    FS_LONG deltaFromStart;
    /* Pointers dependent on destSfnt */
    FS_BYTE *p, *pStart;

    if (t->numberOfContours == 0 )
    {
        return 0; /******/
    }

    deltaFromStart = 0;
    if (AllocateGlyphSpace( _PS_ destSfnt, pos, 10, maxOutSize ))
        return -1;
    pStart = (FS_BYTE *) & (*destSfnt)[pos];
    p = pStart + deltaFromStart; /* Reset pointers */

    if ( t->componentSize > 0 )
    {
        WRITEWORD_INC_MACRO( p, t->componentVersionNumber );
    }
    else
    {
        WRITEWORD_INC_MACRO( p, t->numberOfContours );
    }

    /* Bounding box */
    x = t->xmin;
    WRITEWORD_INC_MACRO( p, x );
    x = t->ymin;
    WRITEWORD_INC_MACRO( p, x );
    x = t->xmax;
    WRITEWORD_INC_MACRO( p, x );
    x = t->ymax;
    WRITEWORD_INC_MACRO( p, x );

    if ( t->componentSize > 0 )
    {
        deltaFromStart = p - pStart;
        if (AllocateGlyphSpace( _PS_ destSfnt, pos, deltaFromStart + t->componentSize * 2, maxOutSize ))
            return -1;
        pStart = (FS_BYTE *) & (*destSfnt)[pos];
        p = pStart + deltaFromStart; /* Reset pointers */
        limit = t->componentSize;
        for ( i = 0; i < limit; i++ )
        {
            x = t->componentData[i];
            WRITEWORD_INC_MACRO( p, x );
        }
        deltaFromStart = p - pStart;
        if (AllocateGlyphSpace(_PS_ destSfnt, pos, deltaFromStart + 2 + t->numberOfInstructions , maxOutSize ))
            return -1;
        pStart = (FS_BYTE *) & (*destSfnt)[pos];
        p = pStart + deltaFromStart; /* Reset pointers */
        if ( t->numberOfInstructions  > 0 )
        {
            limit = t->numberOfInstructions;
            WRITEWORD_INC_MACRO( p, limit );
            f = t->code;
            for ( i = 0; i < limit ; i++ )
            {
                *p++ = f[i];
            }
        }
        return ( p - pStart ); /******/
    }

    deltaFromStart = p - pStart;
    if (AllocateGlyphSpace( _PS_ destSfnt, pos, deltaFromStart + t->numberOfContours * 2, maxOutSize ))
        return -1;
    pStart = (FS_BYTE *) & (*destSfnt)[pos];
    p = pStart + deltaFromStart; /* Reset pointers */
    limit = t->numberOfContours;
    for ( i = 0; i < limit; i++ )
    {
        FS_USHORT ep = t->endPoint[i];
        WRITEWORD_INC_MACRO( p, ep );
    }

    deltaFromStart = p - pStart;
    if (AllocateGlyphSpace( _PS_ destSfnt, pos, deltaFromStart + 2 + t->numberOfInstructions , maxOutSize ))
        return -1;
    pStart = (FS_BYTE *) & (*destSfnt)[pos];
    p = pStart + deltaFromStart; /* Reset pointers */
    limit = t->numberOfInstructions;
    WRITEWORD_INC_MACRO( p, limit );
    f = t->code;
    for ( i = 0; i < limit; i++ )
    {
        *p++ = f[i];
    }

#ifdef FS_MEM_DBG
    STATE.memdbgid = "MTX_Glyph_WriteTTFSpline";
#endif
    f = (FS_BYTE *)FSS_malloc(_PS_ sizeof( FS_BYTE ) * t->numberOfPoints );

    if (STATE.error)
        return -1;

    /* Calculate flags */
    limit = t->numberOfPoints;

    x = y = 0;
    for ( i = 0; i < limit; i++ )
    {
        bitFlags = (FS_BYTE)(t->onCurve[i] & ONCURVE);
        delta = (FS_SHORT)(t->x[i] - x);
        x = t->x[i];
        if ( delta == 0 )
        {
            bitFlags |= NEXT_X_IS_ZERO;
        }
        else if ( delta >= -255 && delta <= 255 )
        {
            bitFlags |= XSHORT;
            if ( delta > 0 )
            {
                bitFlags |= SHORT_X_IS_POS;
            }
        }
        delta = (FS_SHORT)(t->y[i] - y);
        y = t->y[i];
        if ( delta == 0 )
        {
            bitFlags |= NEXT_Y_IS_ZERO;
        }
        else if ( delta >= -255 && delta <= 255 )
        {
            bitFlags |= YSHORT;
            if ( delta > 0 )
            {
                bitFlags |= SHORT_Y_IS_POS;
            }
        }
        f[i] = bitFlags;
    }

    /* limit == t->numberOfPoints already set */
    deltaFromStart = p - pStart;
    if (AllocateGlyphSpace( _PS_ destSfnt, pos, deltaFromStart + limit, maxOutSize ))
    {
        FSS_free(_PS_ f);
        return -1;
    }

    pStart = (FS_BYTE *) & (*destSfnt)[pos];
    p = pStart + deltaFromStart; /* Reset pointers */
    /* Write out bitFlags */
    for ( i = 0; i < limit;)
    {
        FS_BYTE repeat = 0;
        for ( j = (FS_SHORT)(i + 1); j < limit && f[i] == f[j] && repeat < 255; j++ )
            repeat++;
        if ( repeat > 1 )
        {
            *p++ = (FS_BYTE)(f[i] | REPEAT_FLAGS);
            *p++ = repeat;
            i = j;
        }
        else
        {
            *p++ = f[i++];
        }
    }

    /* limit == t->numberOfPoints already set */
    deltaFromStart = p - pStart;
    if (AllocateGlyphSpace( _PS_ destSfnt, pos, deltaFromStart + limit + limit, maxOutSize ))
    {
        FSS_free(_PS_ f);
        return -1;
    }

    pStart = (FS_BYTE *) & (*destSfnt)[pos];
    p = pStart + deltaFromStart; /* Reset pointers */
    /* Write out X */
    x = 0;
    for ( i = 0; i < limit; i++ )
    {
        delta = (FS_SHORT)(t->x[i] - x);
        x = t->x[i];
        if ( f[i] & XSHORT )
        {
            if ( !(f[i] & SHORT_X_IS_POS) ) delta = (FS_SHORT) - delta;
            *p++ = (FS_TINY)delta;
        }
        else if ( !(f[i] & NEXT_X_IS_ZERO ) )
        {
            WRITEWORD_INC_MACRO( p, delta );
        }
    }

    /* Write out Y */
    /* limit == t->numberOfPoints already set */
    deltaFromStart = p - pStart;
    if (AllocateGlyphSpace( _PS_ destSfnt, pos, deltaFromStart + limit + limit, maxOutSize ))
    {
        FSS_free(_PS_ f);
        return -1;
    }
    pStart = (FS_BYTE *) & (*destSfnt)[pos];
    p = pStart + deltaFromStart; /* Reset pointers */

    y = 0;
    for ( i = 0; i < limit; i++ )
    {
        delta = (FS_SHORT)(t->y[i] - y);
        y = t->y[i];
        if ( f[i] & YSHORT )
        {
            if ( !(f[i] & SHORT_Y_IS_POS) ) delta = (FS_SHORT) - delta;
            *p++ = (FS_TINY)delta;
        }
        else if ( !(f[i] & NEXT_Y_IS_ZERO ) )
        {
            WRITEWORD_INC_MACRO( p, delta );
        }
    }

    FSS_free(_PS_ f);
    return ( p - pStart ); /*****/
}


FS_CONST FS_BYTE cvt_pos8                = 255;
FS_CONST FS_BYTE cvt_pos1                = 255 - 7;                    /* == cvt_pos8 - 7 */
FS_CONST FS_BYTE cvt_neg8                = 255 - 7 - 1;                /* == cvt_pos1 - 1 */
FS_CONST FS_BYTE cvt_neg1                = 255 - 7 - 1 - 7;            /* == cvt_neg8 - 7 */
FS_CONST FS_BYTE cvt_neg0                = 255 - 7 - 1 - 7 - 1;        /* == cvt_neg1 - 1 */
FS_CONST FS_BYTE cvt_wordCode            = 255 - 7 - 1 - 7 - 1 - 1;    /* == cvt_neg0 - 1 */

FS_CONST FS_SHORT cvt_lowestCode                     = 255 - 7 - 1 - 7 - 1 - 1;    /* == cvt_wordCode */


/* Reads a FS_SHORT in our 255 FS_SHORT format */
static FS_SHORT Read255Short( FS_BYTE * *pRef )
{
    register FS_SHORT value, code;
    register FS_BYTE *p = *pRef;

    value = *((FS_BYTE *)p++);
    if ( value >= lowestCode )
    {
        code    = value;
        value    = *((FS_BYTE *)p++);    /* value >= 0 && value <= 255 */
        if ( code == wordCode )
        {
            value <<= 8;
            value |= *(FS_BYTE *)p++;
        }
        else if ( code == flipSignCode )
        {
            value++;                            /* value >= 1 && value <= 256 */
            value = (FS_SHORT) - value;                      /* value <= -1 && value >= -256 */
        }
        else
        {
            value = (FS_SHORT)(value + lowestCode + (255 - code) * 256 );
        }
    }
    *pRef = p;
    return value; /*****/
}

/* Emits a TrueType PUSHB[] or NPUSHB[] instruction */
static FS_VOID BytePush( FS_BYTE * *pRef, FS_SHORT argStore[], FS_LONG numberofArgs )
{
    FS_LONG i;
    FS_BYTE *p = *pRef;

    if ( numberofArgs <= 8 )
    {
        *p++ = (FS_BYTE)(0xB0 + numberofArgs - 1); /* PUSHB */
    }
    else
    {
        *p++ = 0x40; /* NPUSHB */
        *p++ = (FS_BYTE)numberofArgs;
    }
    for ( i = 0; i < numberofArgs; i++ )
    {
        *p++ = (FS_BYTE)argStore[i];
    }
    *pRef = p;
}

/* Emits a TrueType PUSHW[] or NPUSHW[] instruction */
static FS_VOID WordPush( FS_BYTE * *pRef, FS_SHORT argStore[], FS_LONG numberofArgs )
{
    FS_LONG i;
    FS_BYTE *p = *pRef;

    if ( numberofArgs <= 8 )
    {
        *p++ = (FS_BYTE)(0xB8 + numberofArgs - 1); /* PUSHW */
    }
    else
    {
        *p++ = 0x41; /* NPUSHW */
        *p++ = (FS_BYTE)numberofArgs;
    }
    for ( i = 0; i < numberofArgs; i++ )
    {
        *p++ = (FS_BYTE)(((FS_USHORT)argStore[i] >> 8) & 0x00ff);
        *p++ = (FS_BYTE)(argStore[i] & 0x00ff);
    }
    *pRef = p;
}


/* Does an optimized sequence of TrueType push statements */
static FS_VOID OptimizingPush( FS_BYTE * *p, FS_SHORT argStore[], FS_LONG numberofArgs )
{
    FS_SHORT i, argument, n2;
    FS_SHORT argCount, n, byteRun, runLength;
    FS_SHORT doBytePush, limit;

    for ( n = 0; numberofArgs > 0;  )
    {
        argCount = (FS_SHORT)(numberofArgs > 255 ? 255 : numberofArgs);
        doBytePush = false;

        limit = 2;
        for ( runLength = 0, i = n; i < (n + argCount); i++ )
        {
            argument = argStore[i];
            if ( argument >= 0  && argument <= 255  )
            {
                /* byteRun = ByteRunLength( &argStore[i], (FS_SHORT)(n + argCount - i) ); */
                /* set 'byteRun' = the number of bytes that all fit within an unsigned byte */
                n2             = (FS_SHORT)(n + argCount - i - 1);
                for ( byteRun = 0; byteRun < n2; )
                {
                    argument = argStore[++byteRun + i];
                    if ( argument > 255 || argument < 0 ) break; /*****/
                }

                /* if we have a run of bytes of length 3 (2 if first or last) or more it is more
                   optimal in terms of space to push them as bytes */
                if ( ((byteRun + i) >= (n + argCount)) )
                {
                    limit = 2;
                }
                if ( byteRun >= limit )
                {
                    if ( runLength > 0 )
                    {
                        argCount    = runLength;
                        doBytePush    = false;
                    }
                    else
                    {
                        argCount    = byteRun;
                        doBytePush    = true;
                    }
                    break; /*****/
                }
            }
            runLength++;
            limit = 3;
        }
        if ( doBytePush )
        {
            BytePush( p, &argStore[n], argCount );
        }
        else
        {
            WordPush( p, &argStore[n], argCount );
        }
        numberofArgs -= argCount;
        n = (FS_SHORT)(n + argCount);
    }
}

/* Reads CTF instructions */
static int ReadAllCTFInstructions( _DS_  Glyph *t, FS_BYTE * *p )
{
    FS_SHORT i, limit, *pushData;
    FS_BYTE uc;
    FS_BYTE *out;

    FSS_free( _PS_ t->pushData);
    t->pushData            = NULL;

    t->pushCount    = limit = Read255UShort( p );

#ifdef FS_MEM_DBG
    STATE.memdbgid = "ReadAllCTFInstructions(pushData)";
#endif
    t->pushData     = (FS_SHORT *)FSS_malloc(_PS_ sizeof( FS_SHORT ) * limit );

    if (STATE.error)
        return (STATE.error);

    pushData = t->pushData;
    for ( i = 0; i < limit; )
    {
        uc = **p;
        if ( uc < lowestCode )
        {
            /* This is a speed optimization */
            pushData[i++] = uc;                        /* Data */
            (*p)++;
        }
        else if ( uc == Hop3Code )
        {
            register FS_SHORT A  = pushData[i - 2];
            (*p)++;                                  /* Hop3Code */
            pushData[i++] = A;                       /* A */
            pushData[i++] = Read255Short( p );       /* X2 */
            pushData[i++] = A;                       /* A */
        }
        else if ( uc == Hop4Code )
        {
            register FS_SHORT A  = pushData[i - 2];
            (*p)++;                                  /* Hop4Code */
            pushData[i++] = A;                       /* A */
            pushData[i++] = Read255Short( p );       /* X2 */
            pushData[i++] = A;                       /* A */
            pushData[i++] = Read255Short( p );       /* X3 */
            pushData[i++] = A;                       /* A */
        }
        else
        {
            pushData[i++] = Read255Short( p );        /* Data */
        }
    }
    t->remainingCodeSize    = limit = Read255UShort( p );    /* limit    == t->remainingCodeSize */

    FSS_free(_PS_ t->code);
    t->code = NULL;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "ReadAllCTFInstructions(code)";
#endif
    t->code = out = (FS_BYTE *)FSS_malloc(_PS_ sizeof( FS_BYTE ) * (i * 3 + limit + 16) );
    /* we know (i*3 + limit + 16 > 0 ... so the test is simple */
    if (STATE.error)
        return (STATE.error);

    OptimizingPush( (FS_BYTE **)&out, pushData, i ); /* Generate TrueType to set the intial stack */

    /* limit == t->remainingCodeSize */
    SYS_MEMCPY( out, *p, (FS_USHORT)limit );
    *p += limit;
    out += limit;
    t->numberOfInstructions = (FS_USHORT)(out - t->code);
    return 0;    /* success */
}

/* Reads a glyph in the ctf format at the location pointed to by <p>. */
/* Returns the number of bytes read */

static FS_LONG ReadCTFSpline( _DS_ register Glyph *t, FS_BYTE *sp )
{
    register FS_LONG i, numberOfPoints;
    FS_SHORT stmp, lastEndPoint;
    FS_SHORT x, y;
    FS_BYTE bitflags;
    FS_BYTE *spStart = sp;
    int status;
    int setBBox = true; /* Initialize here to avoid compiler warnings, Sept 23, 1996 */

    FSS_free( _PS_ t->startPoint);
    t->startPoint    = NULL;
    FSS_free( _PS_ t->onCurve);
    t->onCurve        = NULL;
    FSS_free( _PS_ t->x);
    t->x            = NULL;

    t->numberOfContours = READWORD_INC( &sp );
    if ( t->numberOfContours < 0 )
    {
        t->xmin = READWORD_INC( &sp );
        t->ymin = READWORD_INC( &sp );
        t->xmax = READWORD_INC( &sp );
        t->ymax = READWORD_INC( &sp );
    }
    else
    {
        if ( t->numberOfContours == 0x7fff)   /* flag to read bbox */
        {
            t->numberOfContours = READWORD_INC( &sp );
            t->xmin = READWORD_INC( &sp );
            t->ymin = READWORD_INC( &sp );
            t->xmax = READWORD_INC( &sp );
            t->ymax = READWORD_INC( &sp );
            setBBox = false;
        }
    }

    if ( t->numberOfContours < 0  )
    {
        FS_USHORT flags;
        int weHaveInstructions;

        t->componentVersionNumber = t->numberOfContours;
        do
        {
            flags = (FS_USHORT) READWORD_INC( &sp );
            weHaveInstructions = flags & WE_HAVE_INSTRUCTIONS;
            t->componentData[ t->componentSize++] = flags;

            t->componentData[ t->componentSize++] = READWORD_INC( &sp );
            if ( flags & ARG_1_AND_2_ARE_WORDS )
            {
                /* arg 1 and 2 */
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
            }
            else
            {
                /* arg 1 and 2 as bytes */
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
            }

            if ( flags & WE_HAVE_A_SCALE )
            {
                /* scale */
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
            }
            else if ( flags & WE_HAVE_AN_X_AND_Y_SCALE )
            {
                /* xscale, yscale */
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
            }
            else if ( flags & WE_HAVE_A_TWO_BY_TWO )
            {
                /* xscale, scale01, scale10, yscale */
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
                t->componentData[ t->componentSize++] = READWORD_INC( &sp );
            }
        }
        while ( flags & MORE_COMPONENTS );
        t->numberOfInstructions = 0;
        if ( weHaveInstructions )
        {
            status = ReadAllCTFInstructions( _PS_ t, &sp );
            if (status)
            {
                return -1;
            }
        }
    }
    else
    {
        /* allocate starts and ends as a single piece, 2-nd half is ends */
#ifdef FS_MEM_DBG
        STATE.memdbgid = "ReadCTFSpline(startPoint)";
#endif
        t->startPoint = (FS_USHORT *)FSS_malloc(_PS_ sizeof( FS_USHORT ) * 2 * t->numberOfContours );
        if (STATE.error)
        {
            return -1;
        }

        t->endPoint   = &t->startPoint[t->numberOfContours];

        /* Non Composite */
        lastEndPoint = stmp = 0;
        for ( i = 0; i < t->numberOfContours; i++ )
        {
            t->startPoint[i]    = (FS_USHORT)stmp;
            t->endPoint[i]        = (FS_USHORT)(Read255UShort( &sp ) + lastEndPoint);

            lastEndPoint = t->endPoint[i];
            stmp = (FS_SHORT)(lastEndPoint + 1);
        }
        numberOfPoints = lastEndPoint + 1;
        t->numberOfPoints = (FS_SHORT)numberOfPoints;

#ifdef FS_MEM_DBG
        STATE.memdbgid = "ReadCTFSpline(onCurve)";
#endif
        t->onCurve = (FS_BYTE *)FSS_malloc(_PS_ sizeof( FS_BYTE ) * numberOfPoints );
        if (STATE.error)
        {
            return -1;
        }

        for ( i = 0; i < numberOfPoints; )
        {
            t->onCurve[i++] = *sp++;
        }
        /* allocate x and y together */
#ifdef FS_MEM_DBG
        STATE.memdbgid = "ReadCTFSpline(x)";
#endif
        t->x = (FS_SHORT *)FSS_malloc(_PS_ sizeof( FS_SHORT ) * 2 * numberOfPoints );
        if (STATE.error)
        {
            return -1;
        }

        t->y = &t->x[numberOfPoints];

        x = y = 0;
        for ( i = 0; i < numberOfPoints; i++ )
        {
            register FS_SHORT xBits, yBits;
            FS_USHORT index;
            FS_LONG dx, dy;
            FS_ULONG data, ultmp;
            FS_BYTE xIsNegative, yIsNegative;

            bitflags = t->onCurve[i];

            t->onCurve[i] = (FS_BYTE)(bitflags & 0x80 ? 0 : ONCURVE);
            index = (FS_USHORT)(bitflags & (128 - 1));

            xIsNegative = coordEncoding[index].xIsNegative;
            yIsNegative = coordEncoding[index].yIsNegative;

            /* Determine whether to read 1, 2, or 3 extra bytes */
            /* Each point is represented in a {flag/XCoord/YCoord} triplet  */
            /* X and Y coordinate values can be represented in from 1 to 4 bytes  */
            /* Eg {x 8 bits, y 0 bits}, {x 4 bits, y 4 bits}, {x 16 bits, y 16 bits} */

            /* First byte is for flags which have already been read
               so we are on second byte when we get here
            */

            ultmp = *(FS_BYTE *)sp;  /* read second byte */
            ultmp <<= 24;
            data = ultmp;

            if (coordEncoding[index].byteCount >= 3)
                ultmp = *((FS_BYTE *)sp + 1);
            else
                ultmp = 0;
            ultmp <<= 16;
            data |= ultmp;
            if (coordEncoding[index].byteCount >= 4)
                ultmp = *((FS_BYTE *)sp + 2);
            else
                ultmp = 0;
            ultmp <<= 8;
            data |= ultmp;
            if (coordEncoding[index].byteCount == 5)
                ultmp = *((FS_BYTE *)sp + 3);
            else
                ultmp = 0;

            /*ultmp <<= 0; */
            data |= ultmp;
            sp += coordEncoding[index].byteCount - 1; /* Since 1 byte is for the flags */

            xBits = coordEncoding[index].xBits;
            ultmp = data >> (32 - xBits);
            ultmp &= ((1L << xBits ) - 1);
            dx = ultmp;
            yBits = coordEncoding[index].yBits;
            ultmp = data >> (32 - xBits - yBits);
            ultmp &= ((1L << yBits ) - 1);
            dy = ultmp;
            dx += coordEncoding[index].dxPlus;
            dy += coordEncoding[index].dyPlus;
            if ( xIsNegative ) dx = -dx;
            if ( yIsNegative ) dy = -dy;
            x = (FS_SHORT)(x + dx);
            y = (FS_SHORT)(y + dy);

            t->x[i] = x;
            t->y[i] = y;
        }
        if (setBBox) SetBBox(t); /* Set it, since it is not stored in the CTF format */
        status = ReadAllCTFInstructions( _PS_ t, &sp );
        if (status)
        {
            return -1;
        }
    }
    return sp - spStart;
}

static Glyph *MTX_Glyph_Create( _DS_ FS_BYTE isTTFSpline, FS_BYTE *sp, FS_LONG splineLength, FS_SHORT lsb, FS_SHORT aw,
                                FS_LONG *bytesRead, FS_BYTE *inDataStart, FS_BYTE *inDataEnd )
{
    Glyph *t;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "Glyph";
#endif
    t    = (Glyph *)FSS_malloc(_PS_ sizeof( Glyph ) );

    if (t == NULL)  /* Coverity does not like "if (STATE.error)" */
        return NULL;

    t->advanceWidth          = aw;
    t->leftSideBearing       = lsb;
    t->componentSize        = 0;

    t->numberOfContours      = 0;
    t->numberOfPoints        = 0;
    t->numberOfInstructions  = 0;
    t->startPoint            = NULL;
    t->endPoint              = NULL;
    t->x                     = NULL;
    t->y                     = NULL;
    t->onCurve               = NULL;
    t->code                  = NULL;
    t->componentData         = NULL;
    t->pushData              = NULL;
    t->remainingCode         = NULL;

    t->inDataStart           = inDataStart;
    t->inDataEnd             = inDataEnd;

    t->maxComponentSize = 1024;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "Glyph->componentData";
#endif
    t->componentData = (FS_SHORT *)FSS_malloc( _PS_ sizeof( FS_SHORT ) * t->maxComponentSize );
    if (t->componentData == NULL) /* Coverity does not like "if (STATE.error)" */
    {
        MTX_Glyph_Destroy( _PS_ t);
        return NULL;
    }

    if ( isTTFSpline )
    {
        if ( splineLength )
        {
        }
        else
        {
            *bytesRead = 0;
        }
    }
    else
    {
        if ( splineLength )
        {
            *bytesRead = ReadCTFSpline( _PS_ t, sp );
            if (*bytesRead == -1)
            {
                MTX_Glyph_Destroy( _PS_ t);
                return NULL;
            }
        }
        else
        {
            *bytesRead = 0;
            t->numberOfContours = 0;
        }
    }
    return t; /******/
}


/* It asssumes that spline does point at the beginning of glyph data (IsSpline() == true). */
/* Expands a glyph in the 'glyf' table from the CTF format to the native TTF format on the fly. */

FS_BYTE *MTX_TTC_GetTTFGlyphMemory( _DS_ FS_BYTE *spline, FS_ULONG oldLength, FS_ULONG *newLength)
{
    Glyph *glyph;
    FS_LONG bytesRead, Length, maxOutSize = oldLength + oldLength / 2;
    FS_BYTE *destPtr;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "MTX_TTC_GetTTFGlyphMemory";
#endif
    destPtr = (FS_BYTE *) FSS_malloc(_PS_ (FS_ULONG)maxOutSize);
    if (destPtr == NULL) /* Coverity does not like "if (STATE.error)" */
        return NULL;

    glyph = MTX_Glyph_Create( _PS_ (FS_BYTE)false, spline, oldLength, 0 , 0 , &bytesRead, spline, spline + oldLength );
    if (glyph == 0)
    {
        FSS_free(_PS_ destPtr);
        return NULL;
    }

    /* Recreate TTF glyph data on the fly */
    Length = MTX_Glyph_WriteTTFSpline( _PS_ glyph, &destPtr, 0, &maxOutSize );
    if (Length == -1)
    {
        *newLength = 0;
        FSS_free(_PS_ destPtr);
        MTX_Glyph_Destroy( _PS_ glyph );
        return NULL;
    }
    else
    {
        *newLength = (FS_ULONG)Length;
    }

    MTX_Glyph_Destroy( _PS_ glyph );
    return destPtr;
}

#endif /* FS_ACT3 */
